import React, { useState, useEffect, useRef, useCallback } from 'react';
import { Play, RotateCcw, Trophy, Heart } from 'lucide-react';

interface FallingEmoji {
  id: number;
  emoji: string;
  x: number;
  y: number;
  speed: number;
  caught: boolean;
}

interface CollectingEmoji {
  id: number;
  emoji: string;
  x: number;
  y: number;
  matched: boolean;
}

const EMOJIS = ['🎈', '⭐', '🎯', '🎪', '🎨', '🎭', '🎲', '🎵', '🎸', '🎺'];
const COLLECT_EMOJIS = ['🍎', '🍌', '🍇', '🍓', '🍒', '🥝', '🍑', '🍍'];

export default function EmojiGame() {
  // Game state
  const [gameState, setGameState] = useState<'menu' | 'playing1' | 'playing2' | 'gameOver' | 'completed'>('menu');
  const [level, setLevel] = useState(1);
  const [score, setScore] = useState(0);
  const [lives, setLives] = useState(3);
  const [fallingEmojis, setFallingEmojis] = useState<FallingEmoji[]>([]);
  const [collectingEmojis, setCollectingEmojis] = useState<CollectingEmoji[]>([]);
  const [targetEmoji, setTargetEmoji] = useState('');
  const [collectedCount, setCollectedCount] = useState(0);
  const [targetCount, setTargetCount] = useState(5);
  
  // Refs
  const gameAreaRef = useRef<HTMLDivElement>(null);
  const animationRef = useRef<number>();
  const spawnTimerRef = useRef<number>();

  // Game 1: Falling emojis
  const spawnEmoji = useCallback(() => {
    if (gameState !== 'playing1' || !gameAreaRef.current) return;

    const rect = gameAreaRef.current.getBoundingClientRect();
    const newEmoji: FallingEmoji = {
      id: Date.now() + Math.random(),
      emoji: EMOJIS[Math.floor(Math.random() * EMOJIS.length)],
      x: Math.random() * (rect.width - 60),
      y: -60,
      speed: 2 + level * 0.5,
      caught: false
    };

    setFallingEmojis(prev => [...prev, newEmoji]);
  }, [gameState, level]);

  const moveEmojis = useCallback(() => {
    if (gameState !== 'playing1') return;

    setFallingEmojis(prev => {
      const updated = prev.map(emoji => ({
        ...emoji,
        y: emoji.y + emoji.speed
      }));

      // Check for emojis that reached the bottom
      const hitBottom = updated.filter(emoji => emoji.y > 500 && !emoji.caught);
      if (hitBottom.length > 0) {
        setLives(prevLives => {
          const newLives = prevLives - hitBottom.length;
          if (newLives <= 0) {
            setGameState('gameOver');
          }
          return Math.max(0, newLives);
        });
      }

      // Remove emojis that are off screen
      return updated.filter(emoji => emoji.y < 600 && emoji.y > -100);
    });
  }, [gameState]);

  const handleLineClick = () => {
    if (gameState !== 'playing1') return;

    setFallingEmojis(prev => {
      const bottomEmojis = prev.filter(emoji => emoji.y > 400 && emoji.y < 500 && !emoji.caught);
      if (bottomEmojis.length > 0) {
        setScore(prevScore => prevScore + bottomEmojis.length * 10);
        return prev.map(emoji => 
          bottomEmojis.includes(emoji) 
            ? { ...emoji, caught: true, y: emoji.y - 100, speed: -5 }
            : emoji
        );
      }
      return prev;
    });
  };

  // Game 2: Collecting emojis
  const initializeCollectingGame = () => {
    const shuffled = [...COLLECT_EMOJIS].sort(() => Math.random() - 0.5);
    const target = shuffled[0];
    setTargetEmoji(target);
    setTargetCount(3 + level);
    setCollectedCount(0);

    const emojis: CollectingEmoji[] = [];
    const gridSize = 4;
    const spacing = 80;

    // Add target emojis
    for (let i = 0; i < targetCount; i++) {
      emojis.push({
        id: i,
        emoji: target,
        x: (i % gridSize) * spacing + 50,
        y: Math.floor(i / gridSize) * spacing + 100,
        matched: false
      });
    }

    // Add decoy emojis
    for (let i = 0; i < 8; i++) {
      const decoyEmoji = shuffled[1 + (i % (shuffled.length - 1))];
      emojis.push({
        id: targetCount + i,
        emoji: decoyEmoji,
        x: ((targetCount + i) % gridSize) * spacing + 50,
        y: Math.floor((targetCount + i) / gridSize) * spacing + 100,
        matched: false
      });
    }

    // Shuffle positions
    const shuffledEmojis = emojis.sort(() => Math.random() - 0.5);
    setCollectingEmojis(shuffledEmojis);
  };

  const handleEmojiClick = (clickedEmoji: CollectingEmoji) => {
    if (gameState !== 'playing2' || clickedEmoji.matched) return;

    if (clickedEmoji.emoji === targetEmoji) {
      setCollectingEmojis(prev => 
        prev.map(emoji => 
          emoji.id === clickedEmoji.id ? { ...emoji, matched: true } : emoji
        )
      );
      setCollectedCount(prev => {
        const newCount = prev + 1;
        if (newCount >= targetCount) {
          setScore(prevScore => prevScore + 100);
          setLevel(prevLevel => prevLevel + 1);
          setTimeout(() => {
            if (level >= 5) {
              setGameState('completed');
            } else {
              initializeCollectingGame();
            }
          }, 1000);
        }
        return newCount;
      });
    } else {
      setLives(prev => {
        const newLives = prev - 1;
        if (newLives <= 0) {
          setGameState('gameOver');
        }
        return newLives;
      });
    }
  };

  // Game loop
  useEffect(() => {
    if (gameState === 'playing1') {
      animationRef.current = requestAnimationFrame(function gameLoop() {
        moveEmojis();
        animationRef.current = requestAnimationFrame(gameLoop);
      });

      spawnTimerRef.current = window.setInterval(() => {
        spawnEmoji();
      }, Math.max(1000 - level * 100, 300));

      return () => {
        if (animationRef.current) cancelAnimationFrame(animationRef.current);
        if (spawnTimerRef.current) clearInterval(spawnTimerRef.current);
      };
    }
  }, [gameState, moveEmojis, spawnEmoji]);

  // Check level progression for game 1
  useEffect(() => {
    if (gameState === 'playing1' && score >= level * 100) {
      setLevel(prev => prev + 1);
      if (level >= 3) {
        setGameState('playing2');
        initializeCollectingGame();
      }
    }
  }, [score, level, gameState]);

  const startGame = () => {
    setGameState('playing1');
    setLevel(1);
    setScore(0);
    setLives(3);
    setFallingEmojis([]);
  };

  const resetGame = () => {
    setGameState('menu');
    setLevel(1);
    setScore(0);
    setLives(3);
    setFallingEmojis([]);
    setCollectingEmojis([]);
    setCollectedCount(0);
  };

  return (
    <div className="min-h-screen bg-gradient-to-br from-indigo-500 via-purple-500 to-pink-500 p-4">
      <div className="max-w-md mx-auto bg-white/10 backdrop-blur-lg rounded-3xl p-6 shadow-2xl border border-white/20">
        
        {/* Header */}
        <div className="text-center mb-6">
          <h1 className="text-3xl font-bold text-white mb-2 drop-shadow-lg">
            🎮 Emoji Yakalama
          </h1>
          <div className="flex justify-between items-center text-white/90">
            <div className="flex items-center gap-2">
              <Trophy className="w-5 h-5 text-yellow-300" />
              <span className="font-semibold">{score}</span>
            </div>
            <div className="flex items-center gap-1">
              {[...Array(3)].map((_, i) => (
                <Heart 
                  key={i} 
                  className={`w-5 h-5 ${i < lives ? 'text-red-400 fill-current' : 'text-white/30'}`} 
                />
              ))}
            </div>
            <div className="text-sm font-medium">
              Level {level}
            </div>
          </div>
        </div>

        {/* Game Area */}
        <div 
          ref={gameAreaRef}
          className="relative h-96 bg-black/20 rounded-2xl overflow-hidden border-2 border-white/30"
        >
          {/* Menu */}
          {gameState === 'menu' && (
            <div className="absolute inset-0 flex flex-col items-center justify-center">
              <div className="text-6xl mb-4 animate-bounce">🎯</div>
              <h2 className="text-xl font-bold text-white mb-6 text-center">
                Emojileri yakala!<br />
                <span className="text-sm opacity-80">Çizgiye tıklayarak geri gönder</span>
              </h2>
              <button
                onClick={startGame}
                className="flex items-center gap-2 bg-green-500 hover:bg-green-600 text-white px-6 py-3 rounded-full font-semibold transition-all transform hover:scale-105 shadow-lg"
              >
                <Play className="w-5 h-5" />
                Oyunu Başlat
              </button>
            </div>
          )}

          {/* Game 1: Falling Emojis */}
          {gameState === 'playing1' && (
            <>
              {fallingEmojis.map(emoji => (
                <div
                  key={emoji.id}
                  className={`absolute text-4xl transition-all duration-100 ${
                    emoji.caught ? 'animate-pulse' : ''
                  }`}
                  style={{
                    left: `${emoji.x}px`,
                    top: `${emoji.y}px`,
                    filter: emoji.caught ? 'brightness(1.5)' : 'none'
                  }}
                >
                  {emoji.emoji}
                </div>
              ))}
              
              {/* Catch Line */}
              <div
                onClick={handleLineClick}
                className="absolute bottom-16 left-0 right-0 h-2 bg-gradient-to-r from-yellow-400 to-red-400 cursor-pointer hover:h-3 transition-all shadow-lg animate-pulse"
                style={{ boxShadow: '0 0 20px rgba(255, 255, 0, 0.5)' }}
              />
              
              <div className="absolute bottom-4 left-4 text-white/80 text-sm">
                Emojiler çizgiye gelince tıkla!
              </div>
            </>
          )}

          {/* Game 2: Collecting Emojis */}
          {gameState === 'playing2' && (
            <div className="p-4">
              <div className="text-center mb-4">
                <div className="text-3xl mb-2">{targetEmoji}</div>
                <div className="text-white text-sm">
                  Bu emojiyi topla: {collectedCount}/{targetCount}
                </div>
              </div>
              
              <div className="grid grid-cols-4 gap-2">
                {collectingEmojis.map(emoji => (
                  <button
                    key={emoji.id}
                    onClick={() => handleEmojiClick(emoji)}
                    className={`text-3xl p-2 rounded-lg transition-all transform hover:scale-110 ${
                      emoji.matched 
                        ? 'bg-green-400/50 scale-125' 
                        : 'bg-white/20 hover:bg-white/30'
                    }`}
                    disabled={emoji.matched}
                  >
                    {emoji.emoji}
                  </button>
                ))}
              </div>
            </div>
          )}

          {/* Game Over */}
          {gameState === 'gameOver' && (
            <div className="absolute inset-0 flex flex-col items-center justify-center bg-black/50">
              <div className="text-6xl mb-4">💀</div>
              <h2 className="text-2xl font-bold text-white mb-2">Oyun Bitti!</h2>
              <div className="text-white/80 mb-6">Skor: {score}</div>
              <button
                onClick={resetGame}
                className="flex items-center gap-2 bg-blue-500 hover:bg-blue-600 text-white px-6 py-3 rounded-full font-semibold transition-all transform hover:scale-105"
              >
                <RotateCcw className="w-5 h-5" />
                Tekrar Oyna
              </button>
            </div>
          )}

          {/* Completed */}
          {gameState === 'completed' && (
            <div className="absolute inset-0 flex flex-col items-center justify-center bg-black/50">
              <div className="text-6xl mb-4 animate-bounce">🏆</div>
              <h2 className="text-2xl font-bold text-white mb-2">Tebrikler!</h2>
              <div className="text-white/80 mb-6">Final Skor: {score}</div>
              <button
                onClick={resetGame}
                className="flex items-center gap-2 bg-purple-500 hover:bg-purple-600 text-white px-6 py-3 rounded-full font-semibold transition-all transform hover:scale-105"
              >
                <Play className="w-5 h-5" />
                Yeniden Başla
              </button>
            </div>
          )}
        </div>

        {/* Instructions */}
        <div className="mt-4 text-center text-white/70 text-sm">
          {gameState === 'playing1' && "Emojiler çizgiye gelince tıklayarak yukarı gönder!"}
          {gameState === 'playing2' && "Hedef emojiyi bul ve topla!"}
          {gameState === 'menu' && "Eğlenceli iki aşamalı emoji oyunu"}
        </div>
      </div>
    </div>
  );
}